#/bin/bash
# Copyright (c) 2017 Oracle and/or its affiliates. All rights reserved
#
# ujanke: 2017/08/30 initial release
#
# This script searches for MANIFEST.MF files in Storage Cloud. If a data extraction has been previously completely downloaded
# it will skip those. By this script files will be downloaded into a directory ../export/<Data Extraction Date>. After a
# complete download a file named "download.complete" is being created in that directory. It will be taken as an indicator for a completed download.
# The script reads the content of downloaded MANIFEST.MF file and downloads the zipped data extraction files belonging to this
# data extraction run. As the MANIFEST.MF file lists the MD5 checksum for every single file this script also compares these
# values to validate the correct download.
#
# Please change the value for variable BIContainer below or change this script to pass the container name as a parameter.
#
# Deletion of downloaded data extracts is not handled by this script.
#

export ToolDir=`dirname $0`
export BIContainer=UJ_BIConsole

java -jar ${ToolDir}/ftmcli.jar list --properties-file ${ToolDir}/config/ftmcli.properties ${BIContainer} > logs/fileList
tr -d '\r' < logs/fileList > logs/cleanNames
 
numRows=`cat logs/cleanNames | grep MANIFEST | wc -l`

echo "Number of Rows: ${numRows}"

if [ ${numRows} -eq 0 ]
then
    echo "No data available! Finishing ..."
    exit 255
else
    echo "Running the download of recent BIACM extracts from Storage Cloud for ${numRows} record(s) ..."
fi

if [ ! -d ../export ]
then
    mkdir ../export
fi

rm -f  > logs/manifestDownload.txt logs/fileDownload.txt

for maniFest in `cat logs/cleanNames | grep MANIFEST`
do
    currDocDate=`echo ${maniFest} | awk -F\- '{print $2}' | awk -F\. '{print $1}'`

    if [ ! -d ../export/${currDocDate} ]
    then
    	mkdir ../export/${currDocDate}
    fi

    if [ ! -f ../export/${currDocDate}/download.complete ]
    then
        echo "   >> Saving Files from Extraction Date       : ${currDocDate}"
        echo "   >> Processing Manifest File                : ${maniFest}"

        java -jar ${ToolDir}/ftmcli.jar download --properties-file ${ToolDir}/config/ftmcli.properties ${BIContainer} ${maniFest} > logs/manifestDownload.txt 2>&1

        mv ${maniFest} ../export/${currDocDate}
	
        firstRecord=1
        for fileInfo in `cat ../export/${currDocDate}/${maniFest}`    
        do
            if [ ${firstRecord} -lt 1 ]
            then 
                fileName=`echo ${fileInfo} | awk -F\; '{print $1}'`
                fileChkSum=`echo ${fileInfo} | awk -F\; '{print $2}'`
            
                java -jar ${ToolDir}/ftmcli.jar download --properties-file ${ToolDir}/config/ftmcli.properties ${BIContainer} ${fileName}  > logs/fileDownload.txt 2>&1

                downloadedMD5Sum=`md5sum -b ${fileName} | awk '{print $1}'`
        
                mv ${fileName} ../export/${currDocDate}

                if [ "${downloadedMD5Sum}" = "${fileChkSum}" ]
                then 
                    echo "    >> File ${fileName} downloaded and validated!"            
                else
                    echo "  >> !!! Error when processing file ${fileName}: MD5 chk sums are not identical! Exiting the script!" 
                    echo "      FileName=${fileName}, FileChkSum=${fileChkSum}, downloadedMD5Sum=${downloadedMD5Sum}"
                    exit 1            
                fi
            fi			
            firstRecord=0
            echo "`date +%Y%m%d%H%M%S`" > ../export/${currDocDate}/download.complete
        done
    else
        echo "   >> Skipping Files from Extraction Date ${currDocDate} as previously downloaded on `cat ../export/${currDocDate}/download.complete`"..."
    fi
    echo "-----------------------------------------------------------------------------------------"
    echo ""
done

echo " >> Finished saving Files from Storage Cloud ..."
