import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import java.nio.charset.Charset;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import oracle.ucm.idcws.client.UploadTool;
import oracle.ucm.idcws.client.UploadTool.UploadResults;

import oracle.ucm.idcws.client.bulk.UploadException;

import oracle.ucm.idcws.client.model.content.CheckinSource;
import oracle.ucm.idcws.client.model.content.TransferFile;

import oracle.ucm.idcws.client.model.response.CheckinResponse;

/**
 * Test Class demonstrating programmatic invocation of UploadTool
 * supplying a custom list of items.
 */
public class GenericToolUploadTest
{
	public static void main(String[] args)
	{
		List<String> coreArgs = new ArrayList<String>();
		coreArgs.add("url=https://UpdateThisValue.com/idcws");
		coreArgs.add("policy=oracle/wss_username_token_over_ssl_client_policy");
		coreArgs.add("username=wccadmin");
		coreArgs.add("password=welcome1");
		coreArgs.add("silent=true"); // minimal log output

		List<String> argsList = null;

		argsList = new ArrayList<String>();
		argsList.addAll(coreArgs);
		argsList.add("threads=3");
		argsList.add("throwOnThreadException=false"); // run() will not throw an exception should a thread error

		String[] uploadArgs = argsList.toArray(new String[0]);

		try
		{
			UploadTool uploadTool = new UploadTool();
			try
			{
				// Setup the tool's initial configuration from the supplied arguments.
				boolean terminateEarly = uploadTool.setup(uploadArgs);
				if (terminateEarly)
				{
					return;
				}

				List<CheckinSource> items = new ArrayList<CheckinSource>();
				items.add(new LocalFileSource("ID-001", null, "Test 1", "Public", "", "Document",
					"/C:/Users/mshannon/Desktop/Test/B64Test.java"));
				items.add(new LocalFileSource("ID-002", null, "Test 2", "Public", "", "Document",
					"/C:/Users/mshannon/Desktop/Test/BOMTest.java"));
				items.add(new LocalFileSource("ID-003", null, "Test 3", "Public", "", "Document",
					"/C:/Users/mshannon/Desktop/Test/BooleanTest.java"));
				items.add(new LocalFileSource("ID-004", null, "Test 4", "Public", "", "Document",
					"/C:/Users/mshannon/Desktop/Test/ByteArrayMapTest.java"));
				items.add(new LocalFileSource("ID-005", null, "Test 5", "Public", "", "Document",
					"/C:/Users/mshannon/Desktop/Test/CharBufferTest.java"));

				uploadTool.setCheckinItems(items);

				UploadResults uploadResults = uploadTool.run();
				if (uploadResults != null)
				{
					Map<Integer, Exception> failedCheckins = uploadResults.getAllFailedCheckinsKeyedByTaskNum();
					Map<Integer, UploadException> failedCheckinsDetailed = uploadResults.getFailedCheckinsKeyedByTaskNum();
					for (Map.Entry<Integer, Exception> entry : failedCheckins.entrySet())
					{
						if (failedCheckinsDetailed.containsKey(entry.getKey()))
						{
							UploadException e = failedCheckinsDetailed.get(entry.getKey());
							System.out.println("Checkin with task number " + e.getTaskNumber() + " and identifier "
								+ e.getIdentifier() + " failed with message " + e.getMessage());
						}
						else
						{
							System.out.println("Checkin with task number " + entry.getKey() + " failed with message "
								+ entry.getValue().getMessage());
						}
					}

					Map<Integer, CheckinResponse> successfulCheckins = uploadResults.getSuccessfulCheckinsKeyedByTaskNum();
					for (Map.Entry<Integer, CheckinResponse> entry : successfulCheckins.entrySet())
					{
						CheckinResponse response = entry.getValue();
						System.out.println("Checkin with task number " + response.getTaskNumber() + " and identifier "
							+ response.getIdentifier() + " succeeded. dID=" + response.getDId());
					}
				}

			}
			catch (Exception e)
			{
				throw e;
			}
			finally
			{
				uploadTool.logout();
			}

			System.exit(0);
		}
		catch (Exception e)
		{
			System.exit(1);
		}
	}

	public static class LocalFileSource implements CheckinSource
	{
		private String m_identifier = null;
		private String m_dDocName = null;
		private String m_dDocTitle = null;
		private String m_dSecurityGroup = null;
		private String m_dDocAccount = null;
		private String m_dDocType = null;
		private TransferFile m_primaryFile = null;
		private String m_primaryFileLanguage = null;
		private String m_primaryFileCharacterSet = null;
		private TransferFile m_alternateFile = null;
		private String m_alternateFileLanguage = null;
		private String m_alternateFileCharacterSet = null;

		private static final byte[] s_static_alternate_file_bytes = "See Primary File".getBytes(Charset.forName("UTF-8"));

		public LocalFileSource(
			String identifier,
			String dDocName,
			String dDocTitle,
			String dSecurityGroup,
			String dDocAccount,
			String dDocType,
			String filePath
		) throws IOException
		{
			super();
			this.m_identifier = identifier;
			this.m_dDocName = dDocName;
			this.m_dDocTitle = dDocTitle;
			this.m_dSecurityGroup = dSecurityGroup;
			this.m_dDocAccount = dDocAccount;
			this.m_dDocType = dDocType;
			this.m_primaryFile = new TransferFile(new File(filePath), "application/octet-stream");
		}

		@Override
		public String getIdentifier()
		{
			return m_identifier;
		}

		@Override
		public String getDDocName()
		{
			return m_dDocName;
		}

		@Override
		public String getDDocTitle()
		{
			return m_dDocTitle;
		}

		@Override
		public String getDSecurityGroup()
		{
			return m_dSecurityGroup;
		}

		@Override
		public String getDDocAccount()
		{
			return m_dDocAccount;
		}

		@Override
		public String getDDocType()
		{
			return m_dDocType;
		}

		@Override
		public TransferFile getPrimaryFile()
		{
			return m_primaryFile;
		}

		@Override
		public String getPrimaryFileLanguage()
		{
			return m_primaryFileLanguage;
		}

		@Override
		public String getPrimaryFileCharacterSet()
		{
			return m_primaryFileCharacterSet;
		}

		@Override
		public TransferFile getAlternateFile()
		{
			if (m_alternateFile == null)
			{
				m_alternateFile = new TransferFile(new ByteArrayInputStream(s_static_alternate_file_bytes), "alternate.txt",
					s_static_alternate_file_bytes.length, "text/plain");
			}

			return m_alternateFile;
		}

		@Override
		public String getAlternateFileLanguage()
		{
			return m_alternateFileLanguage;
		}

		@Override
		public String getAlternateFileCharacterSet()
		{
			return m_alternateFileCharacterSet;
		}

		@Override
		public void closePrimaryFile()
		{
			if (m_primaryFile != null)
			{
				try
				{
					InputStream is = m_primaryFile.getInputStream();
					if (is != null)
					{
						is.close();
					}
				}
				catch (IOException ignore)
				{
					;
				}
			}
		}

		@Override
		public void closeAlternateFile()
		{
		}
	}

}
