#!/bin/bash
#
# cloud-init script for Podman compute instance
#
## Copyright (c) 2021, Oracle and/or its affiliates.
## All rights reserved. The Universal Permissive License (UPL), Version 1.0 as shown at http://oss.oracle.com/licenses/upl
#
# Description: Run by cloud-init at instance provisioning.


PGM=$(basename $0)

DOCKER_COMPOSE_REPO="https://github.com/docker/compose/releases/download/v${docker_compose_version}/docker-compose-linux-x86_64"

#######################################
# Print header
# Globals:
#   PGM
#######################################
echo_header() {
  echo "+++ $PGM: $@"
}

#######################################
# Install Podman
#######################################
install_podman() {
	echo_header "Install Podman"

    # Install Podman and related tools on Oracle Linux 9 by installing the container-tools.
	# Includes Podman Docker which enables Podman to work natively with Docker commands.
	sudo dnf install -y container-tools
	sudo podman info

}

#######################################
# Install Docker Compose
#######################################
install_docker_compose() {
	echo_header "Install Docker Compose"
	
	# By default, the latest version of Docker Compose is not available in the 
	# Oracle Linux default repository, so you will need to download it from the Git Hub repository.
	sudo curl -SL $DOCKER_COMPOSE_REPO -o /usr/local/bin/docker-compose
	
	# Set executable permission on the Docker Compose binary
	sudo chmod +x /usr/local/bin/docker-compose

	# Create symbolic link
	sudo ln -s /usr/local/bin/docker-compose /usr/bin/docker-compose
}


#######################################
# Start Podman Socket
#######################################
start_podman_socket() {
	echo_header "Start Podman Socket"
	
	# Enable Docker Compose functionality by creating a Unix socket
	sudo systemctl enable --now podman.socket
	sudo systemctl status podman.socket
}

#######################################
# Install Podman Compose
#######################################
install_podman_compose() {
	echo_header "Install Podman Compose"
	
	# Podman Compose is a Python 3 library that implements the Compose Specification to run with Podman
	sudo dnf install -y python-pip
	#sudo dnf install podman-compose
	sudo -H pip3 install --upgrade pip
	sudo pip3 install podman-compose 
}


#######################################
# Setup docker compose YAML file
# When you create a script from this script, the bash variables are escaped differently.
# Use backslash double dollar sign to escape terraform interpolation.
# See https://faun.pub/terraform-ec2-userdata-and-variables-a25b3859118a
#######################################
setup_docker_compose_yaml() {
	echo_header "Setup docker compose YAML configuration"

	# Make a directory for example Docker Compose file
	sudo mkdir ${mount_dir}

	# Change to folder
	cd ${mount_dir}
	
	# Create docker compose YAML configuration file
	cat <<EOF > docker-compose.yaml
version: '3' 
services: 
  web: 
    image: k8s.gcr.io/echoserver:1.4
    ports:
    	- "${web_server_port}:${web_server_port}"
EOF

}


#######################################
# Main
#######################################
main() {

	cp /etc/motd /etc/motd.bkp
	cat << EOF > /etc/motd
 
I have been modified by cloud-init at $(date)
 
EOF

	install_podman
	install_docker_compose
	start_podman_socket
	install_podman_compose
	setup_docker_compose_yaml
}

main "$@"